#!/usr/bin/env python3
"""
Rename After Effects PNG sequence files from format like:
    NCAAF_SEC_FLA_TD_2_6.png00001
to proper format:
    00001.png
"""

import os
import re
import sys
from pathlib import Path

def rename_png_sequences(directory):
    """
    Rename all PNG sequence files in the given directory and subdirectories.
    """
    directory = Path(directory)
    
    if not directory.exists():
        print(f"Error: Directory '{directory}' does not exist!")
        return
    
    # Pattern to match files like: NCAAF_SEC_FLA_TD_2_6.png00001
    pattern = re.compile(r'^(.+)\.png(\d{5})$')
    
    total_renamed = 0
    
    # Walk through all subdirectories
    for root, dirs, files in os.walk(directory):
        current_dir = Path(root)
        renamed_in_dir = 0
        
        for filename in files:
            match = pattern.match(filename)
            if match:
                # Extract the base name and frame number
                base_name = match.group(1)
                frame_number = match.group(2)
                
                # Create new filename with just frame number
                new_filename = f"{frame_number}.png"
                
                # Full paths
                old_path = current_dir / filename
                new_path = current_dir / new_filename
                
                # Skip if target already exists
                if new_path.exists():
                    print(f"  Skipping {filename} - {new_filename} already exists")
                    continue
                
                # Rename the file
                try:
                    os.rename(old_path, new_path)
                    renamed_in_dir += 1
                except Exception as e:
                    print(f"  Error renaming {filename}: {e}")
        
        if renamed_in_dir > 0:
            print(f"Renamed {renamed_in_dir} files in {current_dir}")
            total_renamed += renamed_in_dir
    
    print(f"\nTotal files renamed: {total_renamed}")

def rename_with_prefix(directory, keep_prefix=True):
    """
    Alternative function that keeps team prefix in filename.
    If keep_prefix=True: NCAAF_SEC_FLA_TD_2_6_00001.png
    """
    directory = Path(directory)
    
    if not directory.exists():
        print(f"Error: Directory '{directory}' does not exist!")
        return
    
    # Pattern to match files like: NCAAF_SEC_FLA_TD_2_6.png00001
    pattern = re.compile(r'^(.+)\.png(\d{5})$')
    
    total_renamed = 0
    
    for root, dirs, files in os.walk(directory):
        current_dir = Path(root)
        renamed_in_dir = 0
        
        for filename in files:
            match = pattern.match(filename)
            if match:
                base_name = match.group(1)
                frame_number = match.group(2)
                
                if keep_prefix:
                    new_filename = f"{base_name}_{frame_number}.png"
                else:
                    new_filename = f"{frame_number}.png"
                
                old_path = current_dir / filename
                new_path = current_dir / new_filename
                
                if new_path.exists():
                    print(f"  Skipping {filename} - {new_filename} already exists")
                    continue
                
                try:
                    os.rename(old_path, new_path)
                    renamed_in_dir += 1
                except Exception as e:
                    print(f"  Error renaming {filename}: {e}")
        
        if renamed_in_dir > 0:
            print(f"Renamed {renamed_in_dir} files in {current_dir}")
            total_renamed += renamed_in_dir
    
    print(f"\nTotal files renamed: {total_renamed}")

def main():
    # Default directory
    default_dir = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/TD_Animation_3_384"
    
    # Get directory from command line or use default
    if len(sys.argv) > 1:
        directory = sys.argv[1]
    else:
        directory = input(f"Enter directory path (or press Enter for default: {default_dir}): ").strip()
        if not directory:
            directory = default_dir
    
    # Ask user for naming preference
    print("\nHow would you like to rename the files?")
    print("1. Simple frame numbers: 00001.png, 00002.png, etc.")
    print("2. Keep team prefix: NCAAF_SEC_FLA_TD_2_6_00001.png, etc.")
    
    choice = input("Enter choice (1 or 2): ").strip()
    
    if choice == "2":
        rename_with_prefix(directory, keep_prefix=True)
    else:
        rename_png_sequences(directory)
    
    print("\nDone!")

if __name__ == "__main__":
    main()